/* Copyright (c) 1995 by Internet Software Consortium
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM DISCLAIMS
 * ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL INTERNET SOFTWARE
 * CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* timers.c - implement timers for the eventlib
 * vix 09sep95 [initial]
 */

#if !defined(LINT) && !defined(CODECENTER)
static const char rcsid[] = "$Id: timers.c,v 1.2 1995/09/17 23:12:25 vixie Exp $";
#endif

#include "eventlib.h"
#include "eventlib_p.h"

static void LinkTimer(evContext_p *ctx, evTimer *old, evTimer *add);
static void UnlinkTimer(evContext_p *ctx, evTimer *old, evTimer *cur);

#ifdef DEBUG
static void PrintTimers(const evContext_p *ctx, char *label);
#else
#define PrintTimers(ctx,label)
#endif

struct timeval
evConsTimeV(int sec, int usec) {
	struct timeval x;

	x.tv_sec = sec;
	x.tv_usec = usec;
	return (x);
}

struct timeval
evAddTimeV(struct timeval addend1, struct timeval addend2) {
	struct timeval x;

	x.tv_sec = addend1.tv_sec + addend2.tv_sec;
	x.tv_usec = addend1.tv_usec + addend2.tv_usec;
	if (x.tv_usec >= 1000000) {
		x.tv_sec++;
		x.tv_usec -= 1000000;
	}
	return (x);
}

struct timeval
evSubTimeV(struct timeval minuend, struct timeval subtrahend) {
	struct timeval x;

	x.tv_sec = minuend.tv_sec - subtrahend.tv_sec;
	if (minuend.tv_usec >= subtrahend.tv_usec) {
		x.tv_usec = minuend.tv_usec - subtrahend.tv_usec;
	} else {
		x.tv_usec = 1000000 - subtrahend.tv_usec + minuend.tv_usec;
		x.tv_sec--;
	}
	return (x);
}

int
evCmpTimeV(struct timeval a, struct timeval b) {
	int x = a.tv_sec - b.tv_sec;

	if (! x)
		x = a.tv_usec - b.tv_usec;
	DPRINTF(99, ("evCmpTimeV(%d.%06d, %d.%06d) -> %d\n",
		    a.tv_sec, a.tv_usec, b.tv_sec, b.tv_usec, x));
	return (x);
}

int
evSetTimer(evContext opaqueCtx,
	   evTimerFunc func,
	   void *uap,
	   struct timeval due,
	   struct timeval inter,
	   evTimerID *opaqueID
) {
	evContext_p *ctx = opaqueCtx.opaque;
	evTimer *id;

	DPRINTF(1, (
"evSetTimer(ctx %#x, func %#x, uap %#x, due %d.%06d, inter %d.%06d)\n",
		    ctx, func, uap,
		    due.tv_sec, due.tv_usec,
		    inter.tv_sec, inter.tv_usec));

	/* due={0,0} is a magic cookie meaning "now." */
	if (due.tv_sec == 0 && due.tv_usec == 0)
		OK(gettimeofday(&due, NULL));

	/* Allocate and fill. */
	OKNEW(id);
	id->func = func;
	id->uap = uap;
	id->due = due;
	id->inter = inter;

	/* Insert in ``due'' order. */
	(void) evRelinkTimers(ctx, /*harmless*/id, id);

	/* Remember the ID if the caller provided us a place for it. */
	if (opaqueID)
		opaqueID->opaque = id;

	return (0);
}

int
evClearTimer(evContext opaqueCtx, evTimerID id) {
	evContext_p *ctx = opaqueCtx.opaque;
	evTimer *del = id.opaque, *old, *cur;

	/* Find element whose ID matches caller's. */
	for (old = NULL, cur = ctx->timers;
	     cur != NULL && cur != del;
	     old = cur, cur = cur->next)
		NULL;
	if (! cur)
		ERR(ENOENT);

	UnlinkTimer(ctx, old, cur);
	(void) free(cur);
	return (0);
}

int
evRelinkTimers(evContext_p *ctx, evTimer *del, evTimer *add) {
	evTimer *old, *cur;

	/* In one pass, delete ``del'' and insert ``add'' in time-due order. */
	for (old = NULL, cur = ctx->timers;
	     cur != NULL && (del != NULL || add != NULL);
	     cur = cur->next) {
		if (cur == del) {
			UnlinkTimer(ctx, old, cur);
			del = NULL;
			continue;
		}
		if (add && evCmpTimeV(cur->due, add->due) > 0) {
			LinkTimer(ctx, old, add);
			cur = add;
			add = NULL;
		}
		old = cur;
	}
	if (add)
		LinkTimer(ctx, old, add);
	if (del)
		ERR(ENOENT);
	PrintTimers(ctx, "after relink");
	return (0);
}

static void
LinkTimer(evContext_p *ctx, evTimer *old, evTimer *add) {
	if (! old) {
		add->next = ctx->timers;
		ctx->timers = add;
	} else {
		add->next = old->next;
		old->next = add;
	}
	PrintTimers(ctx, "after link");
}

static void
UnlinkTimer(evContext_p *ctx, evTimer *old, evTimer *cur) {
	if (! old)
		ctx->timers = cur->next;
	else
		old->next = cur->next;
	PrintTimers(ctx, "after unlink");
}

#ifdef DEBUG
static void PrintTimers(const evContext_p *ctx, char *label) {
	const evTimer *cur;

	DPRINTF(2, ("PrintTimers [%s]:\n", label));
	for (cur = ctx->timers; cur; cur = cur->next)
		DPRINTF(2, (
"\tfunc %#x, uap %#x, due %d.%06d, inter %d.%06d, next %#x\n",
			    cur->func, cur->uap,
			    cur->due.tv_sec, cur->due.tv_usec,
			    cur->inter.tv_sec, cur->inter.tv_usec,
			    cur->next));
}
#endif
